ScreenshotModeOverlay = {}

local ScreenshotModeOverlay_mt = Class(ScreenshotModeOverlay, ScreenElement)

function ScreenshotModeOverlay.new(target, customMt)
    local self = ScreenElement.new(target, customMt or ScreenshotModeOverlay_mt)

    self.SEASON_PERIOD = {
        [1] = g_i18n:getText("earlySpring"),
        [2] = g_i18n:getText("midSpring"),
        [3] = g_i18n:getText("lateSpring"),
        [4] = g_i18n:getText("earlySummer"),
        [5] = g_i18n:getText("midSummer"),
        [6] = g_i18n:getText("lateSummer"),
        [7] = g_i18n:getText("earlyAutumn"),
        [8] = g_i18n:getText("midAutumn"),
        [9] = g_i18n:getText("lateAutumn"),
        [10] = g_i18n:getText("earlyWinter"),
        [11] = g_i18n:getText("midWinter"),
        [12] = g_i18n:getText("lateWinter")
    }

    self.WEATHER = {
        [1] = g_i18n:getText("sun"),
        [2] = g_i18n:getText("rain"),
        [3] = g_i18n:getText("cloudy"),
        [4] = g_i18n:getText("hail")
    }

    self.DAY_TIME = {
        [1] = "00:00",
        [2] = "01:00",
        [3] = "02:00",
        [4] = "03:00",
        [5] = "04:00",
        [6] = "05:00",
        [7] = "06:00",
        [8] = "07:00",
        [9] = "08:00",
        [10] = "09:00",
        [11] = "10:00",
        [12] = "11:00",
        [13] = "12:00",
        [14] = "13:00",
        [15] = "14:00",
        [16] = "15:00",
        [17] = "16:00",
        [18] = "17:00",
        [19] = "18:00",
        [20] = "19:00",
        [21] = "20:00",
        [22] = "21:00",
        [23] = "22:00",
        [24] = "23:00"
    }

    self.WEATHER_DATA = {
        { typeName = "SUN", variation = 1 },
        { typeName = "RAIN", variation = 1 },
        { typeName = "CLOUDY", variation = 1 },
        { typeName = "HAIL", variation = 1 }
    }
    
    return self
end

function ScreenshotModeOverlay:onOpen()
    ScreenshotModeOverlay:superClass().onOpen(self)

    self:backupEnvironment()

    -- Set multiTextOption texts
    self.weather:setTexts(self.WEATHER)
    self.dayTime:setTexts(self.DAY_TIME)
    self.seasonPeriod:setTexts(self.SEASON_PERIOD)

    if not g_currentMission.missionDynamicInfo.isMultiplayer then
        self.pauseGame:setIsChecked(true, true, false) -- Skip animation and doesn't trigger callback (BinaryOptionElement:setIsChecked(isChecked, skipAnimation, forceEvent))
    end

    self.currentWeather = 1 -- required in onSeasonPeriodChanged()
    self.currentDayTime = g_currentMission.environment.currentHour + 1 -- required in onSeasonPeriodChanged()
    
    -- Set default states
    self.seasonPeriod:setState(g_currentMission.environment.currentPeriod, true, false)
    self.weather:setState(self.currentWeather, true, false)
    self.dayTime:setState(self.currentDayTime, true, false)

    if g_localPlayer:getCurrentVehicle() ~= nil then
        local spec = g_localPlayer:getCurrentVehicle().spec_lights
        self.previousBeaconLights = spec:getBeaconLightsVisibility()
        self.previousLightsTypesMask = spec.lightsTypesMask

        self.vehicleCharacter:setIsChecked(true, true, true)
        self.beaconLights:setIsChecked(self.previousBeaconLights, true, true)
        self.defaultLights:setIsChecked(bitAND(self.previousLightsTypesMask, 2^Lights.LIGHT_TYPE_DEFAULT) ~= 0, true, true)
        self.highBeamLights:setIsChecked(bitAND(self.previousLightsTypesMask, 2^Lights.LIGHT_TYPE_HIGHBEAM) ~= 0, true, true)
        self.frontLights:setIsChecked(bitAND(self.previousLightsTypesMask, 2^Lights.LIGHT_TYPE_WORK_FRONT) ~= 0, true, true)
        self.backLights:setIsChecked(bitAND(self.previousLightsTypesMask, 2^Lights.LIGHT_TYPE_WORK_BACK) ~= 0, true, true)
    end

    -- Save previous vars values
    self.previousCamera = g_localPlayer:getCurrentCameraNode()
    self.previousFovY = getFovY(self.previousCamera)

    -- Set up new camera
    self.cameraTarget = createTransformGroup("ScreenshotModeCameraTarget")
    self.camera = createCamera("ScreenshotModeCamera", self.previousFovY, 1, 10000)
    link(getRootNode(), self.cameraTarget)
    link(self.cameraTarget, self.camera)
    setWorldTranslation(self.cameraTarget, getWorldTranslation(self.previousCamera))
    setWorldRotation(self.cameraTarget, getWorldRotation(self.previousCamera))
    g_cameraManager:addCamera(self.camera, nil, false)
    g_cameraManager:setActiveCamera(self.camera)

    -- Camera Rotation by mouse click vars
    self.mouseDragActive = false
    self.lastMousePosX = 0
    self.lastMousePosY = 0
    self.invertCamRotY = self:isCamRotYInverted()

    self.cameraTargetPosX, self.cameraTargetPosY, self.cameraTargetPosZ = getWorldTranslation(self.cameraTarget)
    self.cameraRotX, self.cameraRotY, self.cameraRotZ = getRotation(self.camera)
    self.cameraFov = math.deg(getFovY(self.camera))

    self.blurApplied = false
    self.captureCooldown = 0
    self.captureSound = createSample("captureSound")
    loadSample(self.captureSound, ScreenshotMode.dir .. "sounds/capture-sound.ogg", false)

    self.draw = ScreenshotModeOverlay:superClass().draw

    -- Disable options in multiplayer
    self.pauseGame:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)

    self.seasonPeriod:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.weather:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.dayTime:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)

    self.vehicleCharacter:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.beaconLights:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.defaultLights:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.highBeamLights:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.frontLights:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.backLights:setDisabled(g_currentMission.missionDynamicInfo.isMultiplayer)

    self.vehicleCategory:setVisible(g_localPlayer:getCurrentVehicle() ~= nil)

    self.multiplayerWarning:setVisible(g_currentMission.missionDynamicInfo.isMultiplayer)
    self.unpausedGameWarning:setVisible(false)

    -- Rebuild layout
    self.filters:invalidateLayout()

    -- Add UI blur
    self:setBackgroundBlur(true)

    -- Reset input focus
    self:resetInputFocus()

    if not g_currentMission.missionDynamicInfo.isMultiplayer then
        g_currentMission:setManualPause(true)
    end
end

function ScreenshotModeOverlay:onClickBack()
    if not g_currentMission.missionDynamicInfo.isMultiplayer and self.pauseGame:getIsChecked() then
        self:unpauseGame()
    end

    -- Reset camera
    g_cameraManager:setActiveCamera(self.previousCamera)
    g_cameraManager:removeCamera(self.camera)
    delete(self.camera)
    delete(self.cameraTarget)

    -- Remove UI blur
    self:setBackgroundBlur(false)

    self:changeScreen()
end

function ScreenshotModeOverlay:setBackgroundBlur(value)
    if value then
        if not self.blurApplied then
            g_depthOfFieldManager:pushArea(self.backgroundBlur.absPosition[1], self.backgroundBlur.absPosition[2], self.backgroundBlur.absSize[1], self.backgroundBlur.absSize[2])
        end
    else
        if self.blurApplied then
            g_depthOfFieldManager:popArea()
        end
    end

    self.blurApplied = value
end

function ScreenshotModeOverlay:backupEnvironment()
    self.previousEnvironmentXmlFile = loadXMLFile("environmentXML", ScreenshotMode.dir .. "xml/environment.xml")
    g_currentMission.environment:saveToXMLFile(self.previousEnvironmentXmlFile, "environment")
end

function ScreenshotModeOverlay:unpauseGame()
    -- Reset environment
    g_currentMission.environment:loadFromXMLFile(self.previousEnvironmentXmlFile, "environment")

    -- Reset weather
    -- This console command have a bug: after calling it and as soon as the player opens the Escape menu or the Shop menu, it spawns hail.
    -- The only way to remove it is to reopen the Screenshot Mode. This bug also appears when reloading the weather from EasyDevControl, which also uses this same console command.
    -- For now, I've commented it out, and it seems that it doesn't cause any problems or change anything regarding the game environment restoration.
    --g_currentMission.environment.weather:consoleCommandWeatherReloadData()

    -- Reset season visuals
    g_currentMission.environment:setFixedPeriod(nil)

    -- Update environment
    g_currentMission.environment:update(1, true)
    g_currentMission.environment.lighting:update(9999, true)

    -- Reset vehicle
    local controlledVehicle = g_localPlayer:getCurrentVehicle()
    if controlledVehicle ~= nil then
        if not self.vehicleCharacter:getIsChecked() then
            controlledVehicle:restoreVehicleCharacter()
        end

        controlledVehicle:setBeaconLightsVisibility(self.previousBeaconLights, true, true)
        controlledVehicle:setLightsTypesMask(self.previousLightsTypesMask, true, true)
    end

    -- Unpause game
    g_currentMission:setManualPause(false)
end

function ScreenshotModeOverlay:onClickCapture()
    self:startCooldown()
end

function ScreenshotModeOverlay:startCooldown()
    if self.captureCooldown == 0 then
        self.captureCooldown = 10
    end
end

function ScreenshotModeOverlay:update(dt)
    ScreenshotModeOverlay:superClass().update(self, dt)

    self:translateCam()
    self:rotateCam()
    self:updateFov()

    if Input.isKeyPressed(Input.KEY_return) or Input.isKeyPressed(Input.KEY_f12) or Input.isKeyPressed(Input.KEY_print) then
        self:startCooldown()
    end

    if self.captureCooldown > 0 then
        if self.captureCooldown == 10 then
            if self.getIsVisible(self) then
                self.setVisible(self, false)
                self.setAlpha(self, 0)
                self:setBackgroundBlur(false)
            end
        end

        if self.captureCooldown == 5 then
            if g_screenshotsDirectory ~= nil then
                playSample(self.captureSound, 1, 1, 0, 0, 0)

                local screenshotName = g_screenshotsDirectory .. "fsScreen_" .. getDate("%Y_%m_%d_%H_%M_%S") .. "_ScreenshotMode.png"
                saveScreenshot(screenshotName)
            end
        end

        self.captureCooldown = self.captureCooldown - 1
    end

    if not self.getIsVisible(self) and self.captureCooldown == 0 then
        self.setVisible(self, true)
        self.setAlpha(self, 1)

        if not self.mouseDragActive then
            self:setBackgroundBlur(true)
        end
    end

    if self.captureCooldown > 0 then
        self.draw = ScreenshotModeOverlay:superClass().draw
    elseif self.captureCooldown == 0 and self.mouseDragActive then
        self.draw = self.drawFocusOverlay
    end
end

function ScreenshotModeOverlay:translateCam()
    local speed = 0.175
    local smoothness = 0.175
    local ctrlMultiplier = 0.5
    local shiftMultiplier = 3
    local x, y, z = getWorldTranslation(self.cameraTarget)
    local dirX, dirY, dirZ = 0, 0, 0

    -- Forward / Backward
    if Input.isKeyPressed(Input.KEY_w) then
        dirZ = -1
    elseif Input.isKeyPressed(Input.KEY_s) then
        dirZ = 1
    end

    -- Left / Right
    if Input.isKeyPressed(Input.KEY_a) then
        dirX = -1
    elseif Input.isKeyPressed(Input.KEY_d) then
        dirX = 1
    end

    -- Up / Down
    if Input.isKeyPressed(Input.KEY_q) then
        dirY = 1
    elseif Input.isKeyPressed(Input.KEY_e) then
        dirY = -1
    end

    if not (Input.isKeyPressed(Input.KEY_lctrl) and Input.isKeyPressed(Input.KEY_lshift)) then
        -- Ctrl
        if Input.isKeyPressed(Input.KEY_lctrl) then
            dirZ = dirZ * ctrlMultiplier
            dirX = dirX * ctrlMultiplier
            dirY = dirY * ctrlMultiplier
        end

        -- Shift
        if Input.isKeyPressed(Input.KEY_lshift) then
            dirZ = dirZ * shiftMultiplier
            dirX = dirX * shiftMultiplier
            dirY = dirY * shiftMultiplier
        end
    end

    local lDirX, lDirY, lDirZ = localDirectionToWorld(self.camera, dirX, dirY, dirZ)

    self.cameraTargetPosX = self.cameraTargetPosX + (lDirX * speed)
    self.cameraTargetPosY = self.cameraTargetPosY + (lDirY * speed)
    self.cameraTargetPosZ = self.cameraTargetPosZ + (lDirZ * speed)

    local newX, newY, newZ = MathUtil.vector3Lerp(x, y, z, self.cameraTargetPosX, self.cameraTargetPosY, self.cameraTargetPosZ, smoothness)
    setWorldTranslation(self.cameraTarget, newX, newY, newZ)
end

function ScreenshotModeOverlay:rotateCam()
    local speed = 0.0175
    local smoothness = 0.225
    local x, y, z = getRotation(self.camera)
    local rotZ = 0

    -- Left / Right
    if Input.isKeyPressed(Input.KEY_z) then
        rotZ = 1
    elseif Input.isKeyPressed(Input.KEY_c) then
        rotZ = -1
    end

    self.cameraRotZ = self.cameraRotZ + (rotZ * speed)
    self.cameraRotZ = math.clamp(self.cameraRotZ, math.rad(-45), math.rad(45))

    local newZ = MathUtil.lerp(z, self.cameraRotZ, smoothness)

    if Input.isKeyPressed(Input.KEY_x) then
        newZ = 0
    end

    setRotation(self.camera, x, y, newZ)
end

function ScreenshotModeOverlay:updateFov()
    local speed = 0.0175
    local smoothness = 0.225
    local currentFov = math.deg(getFovY(self.camera))

    local newFov = MathUtil.lerp(currentFov, self.cameraFov, smoothness)
    setFovY(self.camera, math.rad(newFov))
end

function ScreenshotModeOverlay:mouseEvent(posX, posY, isDown, isUp, button, eventUsed)
    ScreenshotModeOverlay:superClass().mouseEvent(self, posX, posY, isDown, isUp, button, eventUsed)

    if isDown then
        local isOnGuiElement = false

        for i = 1, #self:getRootElement().elements do
            local child = self:getRootElement().elements[i]

            local clipX1 = child.absPosition[1]
            local clipY1 = child.absPosition[2]
            local clipX2 = child.absPosition[1] + child.absSize[1]
            local clipY2 = child.absPosition[2] + child.absSize[2]

            if posX >= clipX1 and posX <= clipX2 and posY >= clipY1 and posY <= clipY2 then
                isOnGuiElement = true
            end
        end

        if (Input.isMouseButtonPressed(Input.MOUSE_BUTTON_LEFT) or Input.isMouseButtonPressed(Input.MOUSE_BUTTON_RIGHT)) and not isOnGuiElement then
            self.mouseDragActive = true
            self.lastMousePosX = posX
            self.lastMousePosY = posY

            -- Reset input focus
            self:resetInputFocus()

            self:setBackgroundBlur(false)
            self.draw = self.drawFocusOverlay
        end

        -- Reset FOV
        if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_MIDDLE) then
            setFovY(self.camera, self.previousFovY)
            self.cameraFov = math.deg(self.previousFovY)
        end

        -- Increase / Decrease FOV
        local fovYDeg = math.deg(getFovY(self.camera))
        local fovAdd = 2

        if not (Input.isKeyPressed(Input.KEY_lctrl) and Input.isKeyPressed(Input.KEY_lshift)) then
            if Input.isKeyPressed(Input.KEY_lctrl) then
                fovAdd = 0.5
            end

            if Input.isKeyPressed(Input.KEY_lshift) then
                fovAdd = 4
            end
        end

        if Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_UP) then
            self.cameraFov = math.clamp(self.cameraFov - fovAdd, 1, 100)
        elseif Input.isMouseButtonPressed(Input.MOUSE_BUTTON_WHEEL_DOWN) then
            self.cameraFov = math.clamp(self.cameraFov + fovAdd, 1, 100)
        end
    end

    if isUp then
        if (button == Input.MOUSE_BUTTON_LEFT or button == Input.MOUSE_BUTTON_RIGHT) and (not Input.isMouseButtonPressed(Input.MOUSE_BUTTON_LEFT) and not Input.isMouseButtonPressed(Input.MOUSE_BUTTON_RIGHT)) then
            self.mouseDragActive = false

            self:setBackgroundBlur(true)
            self.draw = ScreenshotModeOverlay:superClass().draw
        end
    end

    if self.mouseDragActive then
        local speed = 3
        local x, y, z = getRotation(self.cameraTarget)
        local dragX = posX - self.lastMousePosX
        local dragY = posY - self.lastMousePosY
        self.lastMousePosX = posX
        self.lastMousePosY = posY

        -- Get input rotation deltas (e.g., from mouse or controller)
        local deltaPitch = (dragY * speed) -- intended X-axis rotation
        local deltaYaw = (dragX * speed)   -- intended Y-axis rotation

        -- Get camera Z rotation in radians
        local camX, camY, camZ = getRotation(self.camera)

        -- Rotate the input axes by the camera Z
        local sinZ = math.sin(camZ)
        local cosZ = math.cos(camZ)

        -- Adjust rotation axes based on camera Z rotation
        local adjustedPitchX = cosZ * deltaPitch
        local adjustedPitchY = -sinZ * deltaPitch

        local adjustedYawX = sinZ * deltaYaw
        local adjustedYawY = cosZ * deltaYaw

        -- Final rotation change applied to transform node
        -- Combine pitch and yaw into a rotation vector
        local finalRotationX = adjustedPitchX + adjustedYawX
        local finalRotationY = adjustedPitchY + adjustedYawY

        if self.invertCamRotY then
            setRotation(self.cameraTarget, x + finalRotationX, y + finalRotationY, z)
        else
            setRotation(self.cameraTarget, x + finalRotationX, y - finalRotationY, z)
        end
    end
end

function ScreenshotModeOverlay:resetInputFocus()
    if FocusManager:getFocusedElement() ~= nil then
        FocusManager:unsetFocus(FocusManager.currentFocusData.focusElement)
        FocusManager:unsetHighlight(FocusManager.currentFocusData.highlightElement)

        FocusManager.currentFocusData.focusElement = nil
        FocusManager.currentFocusData.highlightElement = nil
    end
end

function ScreenshotModeOverlay:drawFocusOverlay()
    local imageOverlay = createImageOverlay(ScreenshotMode.dir .. "images/focus_overlay.dds")
    local x, y = (0.5 - ((1024 / g_screenWidth) / 2)), (0.5 - ((1024 / g_screenHeight) / 2))
    local w, h = (1024 / g_screenWidth), (1024 / g_screenHeight)

    renderOverlay(imageOverlay, x, y, w, h)
end

function ScreenshotModeOverlay:isCamRotYInverted()
    local x, y, z = getWorldRotation(self.previousCamera)

    if z < -1 or z > 1 then
        return true
    else
        return false
    end
end

function ScreenshotModeOverlay:onClickPauseGame(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    if self.pauseGame:getIsChecked() then
        self:backupEnvironment()

        -- Environment options
        self.seasonPeriod:setDisabled(false)
        self.weather:setDisabled(false)
        self.dayTime:setDisabled(false)

        -- Vehicle options
        self.vehicleCharacter:setDisabled(false)
        self.beaconLights:setDisabled(false)
        self.defaultLights:setDisabled(false)
        self.highBeamLights:setDisabled(false)
        self.frontLights:setDisabled(false)
        self.backLights:setDisabled(false)

        self.seasonPeriod:setState(self.seasonPeriod:getState(), true, true)
        self.weather:setState(self.weather:getState(), true, true)
        self.dayTime:setState(self.dayTime:getState(), true, true)

        if g_localPlayer:getCurrentVehicle() ~= nil then
            self.vehicleCharacter:setIsChecked(self.vehicleCharacter:getIsChecked(), true, true)
            self.beaconLights:setIsChecked(self.beaconLights:getIsChecked(), true, true)
            self.defaultLights:setIsChecked(self.defaultLights:getIsChecked(), true, true)
            self.highBeamLights:setIsChecked(self.highBeamLights:getIsChecked(), true, true)
            self.frontLights:setIsChecked(self.frontLights:getIsChecked(), true, true)
            self.backLights:setIsChecked(self.backLights:getIsChecked(), true, true)
        end

        self.unpausedGameWarning:setVisible(false)
        self.filters:invalidateLayout()

        -- Pause game
        g_currentMission:setManualPause(true)
    else
        -- Environment options
        self.seasonPeriod:setDisabled(true)
        self.weather:setDisabled(true)
        self.dayTime:setDisabled(true)

        -- Vehicle options
        self.vehicleCharacter:setDisabled(true)
        self.beaconLights:setDisabled(true)
        self.defaultLights:setDisabled(true)
        self.highBeamLights:setDisabled(true)
        self.frontLights:setDisabled(true)
        self.backLights:setDisabled(true)

        self.unpausedGameWarning:setVisible(true)
        self.filters:invalidateLayout()

        self:unpauseGame()
    end
end

function ScreenshotModeOverlay:onSeasonPeriodChanged(selection)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    g_currentMission.environment:setFixedPeriod(selection)
    g_currentMission.environment:update(1, true)

    self:onWeatherChanged(self.currentWeather)

    -- Increase hour by 1 then decrease it by 1
    -- because environment time is not updated if current day-time is equal to new day-time
    self:onDayTimeChanged(self.currentDayTime + 1)
    self:onDayTimeChanged(self.currentDayTime - 1)
end

function ScreenshotModeOverlay:onWeatherChanged(selection)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    g_currentMission.environment.weather:consoleCommandWeatherSet(self.WEATHER_DATA[selection].typeName, self.WEATHER_DATA[selection].variation)
    g_currentMission.environment:update(1, true)

    self.currentWeather = selection
end

-- Note: Don't use consoleCommandSetDayTime() because it increments days if we decrease the time and this make clouds buggy
function ScreenshotModeOverlay:onDayTimeChanged(selection)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    -- Clamp to 24 (index)
    -- Useful when calling self:onDayTimeChanged(self.currentDayTime + 1) and self.currentDayTime is already equal to 24
    if selection > 24 then
        selection = 1
    end
    
    -- Array index starts at 1, hours are indexed from 0
    local realHour = selection - 1

    local env = g_currentMission.environment

    env:setEnvironmentTime(env.currentMonotonicDay, env.currentDay, math.floor(realHour * 1000 * 60 * 60), env.daysPerPeriod, false)
    env:update(1, true)

    self:onWeatherChanged(self.currentWeather)
    env.lighting:update(9999, true)

    self.currentDayTime = selection
end

function ScreenshotModeOverlay:onClickVehicleCharacter(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    local controlledVehicle = g_localPlayer:getCurrentVehicle()

    if controlledVehicle ~= nil then
        if self.vehicleCharacter:getIsChecked() then
            controlledVehicle:restoreVehicleCharacter()
        else
            controlledVehicle:deleteVehicleCharacter()
        end
    end
end

function ScreenshotModeOverlay:onClickBeaconLights(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    local controlledVehicle = g_localPlayer:getCurrentVehicle()

    if controlledVehicle ~= nil then
        controlledVehicle.spec_lights:setBeaconLightsVisibility(self.beaconLights:getIsChecked(), true, true)
    end
end

function ScreenshotModeOverlay:onClickDefaultLights(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    -- Note: "state" returns 1 and 2 for false and true. Use getIsChecked() to get 0 and 1 values.

    local controlledVehicle = g_localPlayer:getCurrentVehicle()

    if controlledVehicle ~= nil then
        local spec = controlledVehicle.spec_lights

        if self.defaultLights:getIsChecked() ~= (bitAND(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_DEFAULT) ~= 0) then
            local lightsTypesMask = bitXOR(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_DEFAULT)
            controlledVehicle:setLightsTypesMask(lightsTypesMask, true, true)
        end
    end
end

function ScreenshotModeOverlay:onClickHighBeamLights(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    local controlledVehicle = g_localPlayer:getCurrentVehicle()

    if controlledVehicle ~= nil then
        local spec = controlledVehicle.spec_lights

        if self.highBeamLights:getIsChecked() ~= (bitAND(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_HIGHBEAM) ~= 0) then
            local lightsTypesMask = bitXOR(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_HIGHBEAM)
            controlledVehicle:setLightsTypesMask(lightsTypesMask, true, true)
        end
    end
end

function ScreenshotModeOverlay:onClickFrontLights(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    local controlledVehicle = g_localPlayer:getCurrentVehicle()

    if controlledVehicle ~= nil then
        local spec = controlledVehicle.spec_lights

        if self.frontLights:getIsChecked() ~= (bitAND(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_WORK_FRONT) ~= 0) then
            local lightsTypesMask = bitXOR(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_WORK_FRONT)
            controlledVehicle:setLightsTypesMask(lightsTypesMask, true, true)
        end
    end
end

function ScreenshotModeOverlay:onClickBackLights(state)
    if g_currentMission.missionDynamicInfo.isMultiplayer then
        return
    end

    local controlledVehicle = g_localPlayer:getCurrentVehicle()

    if controlledVehicle ~= nil then
        local spec = controlledVehicle.spec_lights
        
        if self.backLights:getIsChecked() ~= (bitAND(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_WORK_BACK) ~= 0) then
            local lightsTypesMask = bitXOR(spec.lightsTypesMask, 2^Lights.LIGHT_TYPE_WORK_BACK)
            controlledVehicle:setLightsTypesMask(lightsTypesMask, true, true)
        end
    end
end
