------------------------------------------------------------
-- Fast Discharge + Capacity Boost - FS25 (SINGLE TOGGLE)
-- Toggle BOTH: ALT + B
-- Discharge Multiplier: ALT + N
-- Capacity Multiplier: ALT + M
------------------------------------------------------------

FastDischarge = {}

------------------------------------------------------------
-- MULTIPLIERS
------------------------------------------------------------

FastDischarge.dischargeLevels = { 1.0, 1.25, 2.0, 2.5, 3.0, 5.0, 10.0 }
FastDischarge.capacityLevels  = { 1.0, 1.25, 2.0, 2.5, 3.0, 5.0, 10.0 }

------------------------------------------------------------
-- HELPER: FORMAT NUMBER FOR HUD (e.g. 20000 -> "20,000")
------------------------------------------------------------

function FastDischarge.formatNumber(num)
    local formatted = tostring(math.floor(num + 0.5))
    local k
    while true do
        formatted, k = formatted:gsub("^(-?%d+)(%d%d%d)", "%1,%2")
        if k == 0 then break end
    end
    return formatted
end

------------------------------------------------------------
-- HELPER: GET BOOSTED CAPACITY FOR HUD (ORIGINAL * MULTIPLIER)
-- Does NOT modify any game values, purely for display.
------------------------------------------------------------

function FastDischarge.getBoostedCapacity(vehicle)
    if vehicle == nil or vehicle.spec_fillUnit == nil then
        return nil
    end

    local specFD = vehicle.spec_fastDischarge
    if specFD == nil then
        return nil
    end

    local fillSpec = vehicle.spec_fillUnit
    local fillUnits = fillSpec.fillUnits
    if fillUnits == nil then
        return nil
    end

    -- Find the first fillUnit we have tagged with _fdOriginalCapacity
    local originalCap = nil
    for _, unit in ipairs(fillUnits) do
        if unit._fdOriginalCapacity ~= nil then
            originalCap = unit._fdOriginalCapacity
            break
        end
    end

    -- Fallback: if not tagged yet, try capacity of first fillUnit
    if originalCap == nil and fillUnits[1] ~= nil then
        originalCap = fillUnits[1].capacity or 0
    end

    if originalCap == nil then
        return nil
    end

    local mult = FastDischarge.capacityLevels[specFD.cLevel or 1] or 1.0
    return originalCap * mult
end

------------------------------------------------------------
-- SPECIALIZATION CHECK
------------------------------------------------------------

function FastDischarge.prerequisitesPresent(specializations)
    return SpecializationUtil.hasSpecialization(Dischargeable, specializations)
end

------------------------------------------------------------
-- EVENT REGISTRATION
------------------------------------------------------------

function FastDischarge.registerEventListeners(vehicleType)
    SpecializationUtil.registerEventListener(vehicleType, "onLoad", FastDischarge)
    SpecializationUtil.registerEventListener(vehicleType, "onDraw", FastDischarge)
    SpecializationUtil.registerEventListener(vehicleType, "onRegisterActionEvents", FastDischarge)
end

function FastDischarge.registerFunctions(vehicleType)
    SpecializationUtil.registerFunction(vehicleType, "setFastDischargeState", FastDischarge.setFastDischargeState)
end

------------------------------------------------------------
-- VEHICLE INIT
------------------------------------------------------------

function FastDischarge:onLoad(savegame)
    self.spec_fastDischarge = {
        active = false,  -- shared toggle for BOTH discharge + capacity
        dLevel = 1,      -- discharge multiplier index
        cLevel = 1,      -- capacity multiplier index
        events = {},

        lastFillLevel = 0,
        lastFillTime  = g_time or 0,
        liveLps       = 0
    }

    -- Ensure vanilla capacity on load
    FastDischarge.applyCapacityBoost(self)
end

------------------------------------------------------------
-- LIVE DISCHARGE RATE
------------------------------------------------------------

function FastDischarge:updateLiveRate(dt)
    local specFD = self.spec_fastDischarge
    local specF  = self.spec_fillUnit
    if specFD == nil or specF == nil or specF.fillUnits == nil then
        return
    end

    local fillLevel = 0
    for _, unit in ipairs(specF.fillUnits) do
        fillLevel = fillLevel + (unit.fillLevel or 0)
    end

    local now      = g_time or 0
    local lastTime = specFD.lastFillTime or now
    local timeDiff = now - lastTime

    if timeDiff > 120 then
        local lastFill = specFD.lastFillLevel or fillLevel
        local delta    = lastFill - fillLevel

        if delta > 0 then
            specFD.liveLps = (delta / timeDiff) * 1000
        else
            specFD.liveLps = 0
        end

        specFD.lastFillLevel = fillLevel
        specFD.lastFillTime  = now
    end
end

------------------------------------------------------------
-- HUD
------------------------------------------------------------

function FastDischarge:onDraw(dt)
    local spec = self.spec_fastDischarge
    if spec == nil then return end

    FastDischarge.updateLiveRate(self, dt)

    local dMult = FastDischarge.dischargeLevels[spec.dLevel] or 1.0
    local cMult = FastDischarge.capacityLevels[spec.cLevel] or 1.0
    local state = spec.active and "ON" or "OFF"

    g_currentMission:addExtraPrintText(
        string.format("Fast Discharge: x%.2f (%s)", dMult, state)
    )

    g_currentMission:addExtraPrintText(
        string.format("Capacity: x%.2f (%s)", cMult, state)
    )

    g_currentMission:addExtraPrintText(
        string.format("Discharge Rate: %.0f L/s", spec.liveLps or 0)
    )

    --------------------------------------------------------
    -- NEW: STATIC HUD READOUT OF NEW CAPACITY
    -- (based on original XML capacity × selected multiplier)
    --------------------------------------------------------
    local boostedCap = FastDischarge.getBoostedCapacity(self)
    if boostedCap ~= nil then
        g_currentMission:addExtraPrintText(
            string.format("New Capacity: %s L", FastDischarge.formatNumber(boostedCap))
        )
    end
end

------------------------------------------------------------
-- INPUT REGISTRATION
------------------------------------------------------------

function FastDischarge:onRegisterActionEvents(isActiveForInput, isActiveForInputIgnoreSelection)
    if not self.isClient then return end

    local spec = self.spec_fastDischarge
    self:clearActionEventsTable(spec.events)

    if self:getIsActiveForInput(true, true) then
        -- ALT + B : shared toggle
        local _, id1 = self:addActionEvent(
            spec.events, "FASTDISCHARGE_TOGGLE",
            self, FastDischarge.toggleBoost,
            false, true, false, true
        )
        g_inputBinding:setActionEventTextPriority(id1, GS_PRIO_HIGH)

        -- ALT + N : discharge multiplier
        local _, id2 = self:addActionEvent(
            spec.events, "FASTDISCHARGE_NEXT_LEVEL",
            self, FastDischarge.nextDischargeLevel,
            false, true, false, true
        )
        g_inputBinding:setActionEventTextPriority(id2, GS_PRIO_HIGH)

        -- ALT + M : capacity multiplier
        local _, id3 = self:addActionEvent(
            spec.events, "FASTDISCHARGE_NEXT_CAPACITY",
            self, FastDischarge.nextCapacityLevel,
            false, true, false, true
        )
        g_inputBinding:setActionEventTextPriority(id3, GS_PRIO_HIGH)
    end
end

------------------------------------------------------------
-- INPUT CALLBACKS
------------------------------------------------------------

function FastDischarge:toggleBoost()
    local spec = self.spec_fastDischarge
    if spec == nil then return end

    self:setFastDischargeState(not spec.active)

    -- Apply / restore capacity when toggling
    FastDischarge.applyCapacityBoost(self)
end

function FastDischarge:nextDischargeLevel()
    local spec = self.spec_fastDischarge
    if spec == nil then return end

    spec.dLevel = spec.dLevel + 1
    if spec.dLevel > #FastDischarge.dischargeLevels then
        spec.dLevel = 1
    end
end

function FastDischarge:nextCapacityLevel()
    local spec = self.spec_fastDischarge
    if spec == nil then return end

    spec.cLevel = spec.cLevel + 1
    if spec.cLevel > #FastDischarge.capacityLevels then
        spec.cLevel = 1
    end

    -- Only apply immediately if the system is ON
    if spec.active then
        FastDischarge.applyCapacityBoost(self)
    end
end

------------------------------------------------------------
-- STATE SETTER
------------------------------------------------------------

function FastDischarge:setFastDischargeState(active)
    local spec = self.spec_fastDischarge
    if spec == nil then return end
    spec.active = active
end

------------------------------------------------------------
-- FUEL-SAFE CAPACITY BOOST
-- Applies ONLY to fillUnits referenced by dischargeNodes
------------------------------------------------------------

function FastDischarge.applyCapacityBoost(vehicle)
    if vehicle == nil then return end

    local specFD   = vehicle.spec_fastDischarge
    local fillSpec = vehicle.spec_fillUnit
    local specD    = vehicle.spec_dischargeable

    if specFD == nil or fillSpec == nil or fillSpec.fillUnits == nil then
        return
    end

    --------------------------------------------------------
    -- Build set of "target" fill units from dischargeNodes
    --------------------------------------------------------
    local targetUnits = {}

    if specD ~= nil and specD.dischargeNodes ~= nil then
        for _, node in ipairs(specD.dischargeNodes) do
            if node.fillUnitIndex ~= nil then
                targetUnits[node.fillUnitIndex] = true
            end
        end
    end

    -- If there are no dischargeNodes, do nothing
    if next(targetUnits) == nil then
        return
    end

    --------------------------------------------------------
    -- Apply capacity boost ONLY to those target fill units
    --------------------------------------------------------
    for index, unit in ipairs(fillSpec.fillUnits) do

        -- Skip fuel, DEF, etc. (they are not in targetUnits)
        if targetUnits[index] then

            if unit._fdOriginalCapacity == nil then
                unit._fdOriginalCapacity     = unit.capacity
                unit._fdOriginalMassPerLiter = unit.massPerLiter
            end

            if specFD.active then
                local mult   = FastDischarge.capacityLevels[specFD.cLevel] or 1.0
                local newCap = unit._fdOriginalCapacity * mult
                unit.capacity = math.max(newCap, unit.fillLevel or 0)

                -- Prevent mass scaling with capacity
                unit.updateMass = false
            else
                -- Restore vanilla values
                unit.capacity     = math.max(unit._fdOriginalCapacity, unit.fillLevel or 0)
                unit.massPerLiter = unit._fdOriginalMassPerLiter
                unit.updateMass   = true
            end
        end
    end
end

------------------------------------------------------------
-- DISCHARGE SPEED OVERRIDE
------------------------------------------------------------

local originalSetCurrentDischargeNodeIndex =
    Dischargeable.setCurrentDischargeNodeIndex

function Dischargeable:setCurrentDischargeNodeIndex(index, ...)
    local vehicle = self
    local specFD  = vehicle.spec_fastDischarge
    local specD   = vehicle.spec_dischargeable

    if specD ~= nil and specD.dischargeNodes ~= nil then

        -- Cache original emptySpeed once
        for _, node in ipairs(specD.dischargeNodes) do
            if node._fdOriginalEmptySpeed == nil then
                node._fdOriginalEmptySpeed = node.emptySpeed
            end
        end

        if specFD ~= nil and specFD.active then
            local mult = FastDischarge.dischargeLevels[specFD.dLevel] or 1.0
            for _, node in ipairs(specD.dischargeNodes) do
                node.emptySpeed = node._fdOriginalEmptySpeed * mult
            end
        else
            for _, node in ipairs(specD.dischargeNodes) do
                node.emptySpeed = node._fdOriginalEmptySpeed
            end
        end
    end

    return originalSetCurrentDischargeNodeIndex(self, index, ...)
end
