-- FilltypeExtension.lua
-- Version: 1.0.0.0
-- Author:  Tidde


local FilltypeExtension = rawget(_G, "g_FilltypeExtensionStationaryBalerFW") or {}
if FilltypeExtension._initialized then return end
_G.g_FilltypeExtensionStationaryBalerFW = FilltypeExtension
FilltypeExtension._initialized = true


FilltypeExtension.originalStationaryFillTypes            = "STRAW DRYGRASS_WINDROW GRASS_WINDROW CHAFF SUGARBEET_CUT FORAGE"
FilltypeExtension.Filltypes                              = "STRAW GRASS_WINDROW DRYGRASS_WINDROW CHAFF SUGARBEET_CUT FORAGE ALFALFA_WINDROW CLOVER_WINDROW"
FilltypeExtension.additiveFilltypes                      = "GRASS_WINDROW DRYGRASS_WINDROW CHAFF ALFALFA_WINDROW CLOVER_WINDROW"
FilltypeExtension.additiveFilltypesForageWagon           = "GRASS_WINDROW ALFALFA_WINDROW CLOVER_WINDROW"
FilltypeExtension.additiveFilltypesForageHarvester       = "GRASS_WINDROW DRYGRASS_WINDROW CHAFF ALFALFA_WINDROW CLOVER_WINDROW"

FilltypeExtension.requireBaleType = true   -- true: nur FillTypes, die auch BaleTypes haben 
FilltypeExtension.logMissing      = false

FilltypeExtension._origStatUpper = (FilltypeExtension.originalStationaryFillTypes or ""):upper()


local function ftIndex(name)
    return (name and name ~= "") and g_fillTypeManager:getFillTypeIndexByName(name:upper()) or nil
end

local function toNameList(value)
    if type(value) == "table" then
        return value
    elseif type(value) == "string" then
        local out = {}
        for token in value:gmatch("%S+") do
            out[#out+1] = token
        end
        return out
    end
    return {}
end

local function hasBaleTypeEligible(ftIdx)
    if not FilltypeExtension.requireBaleType then return true end
    if g_baleTypeManager and g_baleTypeManager.fillTypeToBaleTypes then
        local list = g_baleTypeManager.fillTypeToBaleTypes[ftIdx]
        return list ~= nil and #list > 0
    end
    return true
end

local function buildFiltered(list, label)
    local names = toNameList(list)
    local okNames, okIdx, missing = {}, {}, {}
    for _, name in ipairs(names) do
        local idx = ftIndex(name)
        if idx ~= nil and hasBaleTypeEligible(idx) then
            local up = name:upper()
            okNames[#okNames+1] = up
            okIdx[#okIdx+1] = idx
        else
            missing[#missing+1] = tostring(name)
        end
    end
    local str = table.concat(okNames, " ")
    if FilltypeExtension.logMissing and #missing > 0 then
        print(("[FilltypeExtension/FW] Ignored %s (not registered%s): %s"):format(
            label, FilltypeExtension.requireBaleType and " / no BaleType" or "", table.concat(missing, ", ")))
    end
    return str, okIdx
end

local function buildFilteredOnce()
    if FilltypeExtension._filteredBuilt then return end
    FilltypeExtension._filteredBuilt = true

    FilltypeExtension._intakeStr, FilltypeExtension._intakeIdx = buildFiltered(FilltypeExtension.Filltypes,                        "Stationary-Baler Intake")
    FilltypeExtension._addBalStr, FilltypeExtension._addBalIdx = buildFiltered(FilltypeExtension.additiveFilltypes,                "Baler Additives")
    FilltypeExtension._addFWStr,  FilltypeExtension._addFWIdx  = buildFiltered(FilltypeExtension.additiveFilltypesForageWagon,     "ForageWagon Additives")
    FilltypeExtension._addFHStr,  FilltypeExtension._addFHIdx  = buildFiltered(FilltypeExtension.additiveFilltypesForageHarvester, "ForageHarvester Additives")

    if FilltypeExtension._intakeStr == "" then
        FilltypeExtension._intakeStr = FilltypeExtension.originalStationaryFillTypes or ""
        FilltypeExtension._intakeIdx = {}
    end
end

local function hasSpec(self, specName)
    local cls = g_specializationManager:getSpecializationByName(specName)
    return cls ~= nil and SpecializationUtil.hasSpecialization(cls, self.specializations)
end


local function onLoadFillUnitFromXML(self, superFunc, xmlFile, key, entry, index)
    local vehType = (xmlFile:getValue("vehicle#type","") or ""):upper()
    if vehType == "BALERSTATIONARY" then
        buildFilteredOnce()
        local cur = xmlFile:getValue(key.."#fillTypes")
        if cur ~= nil and cur:upper() == FilltypeExtension._origStatUpper then
            xmlFile:setValue(key.."#fillTypes", FilltypeExtension._intakeStr)
        end
    end
    return superFunc(self, xmlFile, key, entry, index)
end
FillUnit.loadFillUnitFromXML = Utils.overwrittenFunction(FillUnit.loadFillUnitFromXML, onLoadFillUnitFromXML)

local function onLoadSpecValueFillTypes(xmlFile, superFunc, customEnvironment, baseDir)
    local vehType = (xmlFile:getValue("vehicle#type","") or ""):upper()
    if vehType == "BALERSTATIONARY" then
        buildFilteredOnce()
        local root = xmlFile:getRootName()
        local i = 0
        while true do
            local cfgKey = ("%s.fillUnit.fillUnitConfigurations.fillUnitConfiguration(%d)"):format(root, i)
            if not xmlFile:hasProperty(cfgKey) then break end
            local j = 0
            while true do
                local unitKey = ("%s.fillUnits.fillUnit(%d)"):format(cfgKey, j)
                if not xmlFile:hasProperty(unitKey) then break end
                local showInShop = xmlFile:getValue(unitKey.."#showInShop")
                local capacity   = xmlFile:getValue(unitKey.."#capacity")
                if (showInShop == nil or showInShop) and (capacity == nil or capacity > 0) then
                    local cur = xmlFile:getValue(unitKey.."#fillTypes")
                    if cur ~= nil and cur:upper() == FilltypeExtension._origStatUpper then
                        xmlFile:setValue(unitKey.."#fillTypes", FilltypeExtension._intakeStr)
                    end
                end
                j = j + 1
            end
            i = i + 1
        end
    end
    return superFunc(xmlFile, customEnvironment, baseDir)
end
FillUnit.loadSpecValueFillTypes = Utils.overwrittenFunction(FillUnit.loadSpecValueFillTypes, onLoadSpecValueFillTypes)


local function balerOnLoad(self, superFunc, savegame)
    if self.xmlFile then
        local vehType = (self.xmlFile:getValue("vehicle#type","") or ""):upper()
        if vehType == "BALERSTATIONARY" then
            buildFilteredOnce()
            local cur = self.xmlFile:getValue("vehicle.baler.additives#fillTypes")
            if cur ~= nil and (cur:upper() == "GRASS_WINDROW CHAFF" or cur:upper() == "GRASS_WINDROW") then
                self.xmlFile:setValue("vehicle.baler.additives#fillTypes", FilltypeExtension._addBalStr)
            end
        end
    end

    local ret = superFunc(self, savegame)

    if hasSpec(self, "BalerStationary") then
        self.spec_baler = self.spec_baler or {}
        if self.spec_baler.additives == nil then
            self.spec_baler.additives = { enabled = false }
        elseif self.spec_baler.additives.enabled == nil then
            self.spec_baler.additives.enabled = false
        end
        if self.spec_baler.variableSpeedLimit == nil then
            self.spec_baler.variableSpeedLimit = { enabled = false }
        elseif self.spec_baler.variableSpeedLimit.enabled == nil then
            self.spec_baler.variableSpeedLimit.enabled = false
        end
    end

    return ret
end
Baler.onLoad = Utils.overwrittenFunction(Baler.onLoad, balerOnLoad)

local function balerOnPostLoad(self, superFunc, savegame)
    local ret = superFunc(self, savegame)
    if not hasSpec(self, "BalerStationary") then return ret end

    buildFilteredOnce()

    local function allowAll(tbl)
        if not tbl then return end
        for _, idx in ipairs(FilltypeExtension._intakeIdx or {}) do
            tbl[idx] = true
        end
    end

    local spec = self.spec_baler
    if spec then
        allowAll(spec.allowedFillTypes)
        allowAll(spec.supportedFillTypes)
        allowAll(spec.pickUpFillTypes)
    end

    local specStat = self.spec_balerStationary or self.spec_balerstationary
    if specStat and specStat.supportedFillTypes then
        allowAll(specStat.supportedFillTypes)
    end

    return ret
end
Baler.onPostLoad = Utils.overwrittenFunction(Baler.onPostLoad, balerOnPostLoad)

local function forageWagonOnLoad(self, superFunc, savegame)
    if self.xmlFile then
        local vehType = (self.xmlFile:getValue("vehicle#type","") or ""):upper()
        if vehType == "FORAGEWAGON" then
            buildFilteredOnce()
            if self.xmlFile:hasProperty("vehicle.forageWagon.additives") then
                local cur = self.xmlFile:getValue("vehicle.forageWagon.additives#fillTypes")
                if cur ~= nil then
                    self.xmlFile:setValue("vehicle.forageWagon.additives#fillTypes", FilltypeExtension._addFWStr)
                end
            end
        end
    end

    local ret = superFunc(self, savegame)

    if self.spec_forageWagon and self.spec_forageWagon.additives and self.spec_forageWagon.additives.enabled == nil then
        self.spec_forageWagon.additives.enabled = false
    end

    return ret
end
ForageWagon.onLoad = Utils.overwrittenFunction(ForageWagon.onLoad, forageWagonOnLoad)

local function combineLoadCombineSetup(self, superFunc, xmlFile, baseKey, entry)
    local vehType = (xmlFile:getValue("vehicle#type","") or ""):upper()
    if vehType == "COMBINEDRIVABLE" or vehType == "COMBINE" then
        buildFilteredOnce()
        xmlFile:setValue("vehicle.combine.additives#fillTypes", FilltypeExtension._addFHStr)
    end
    return superFunc(self, xmlFile, baseKey, entry)
end
Combine.loadCombineSetup = Utils.overwrittenFunction(Combine.loadCombineSetup, combineLoadCombineSetup)

local function combineOnLoad(self, superFunc, savegame)
    local ret = superFunc(self, savegame)
    if self.spec_combine and self.spec_combine.additives and self.spec_combine.additives.enabled == nil then
        self.spec_combine.additives.enabled = false
    end
    return ret
end
Combine.onLoad = Utils.overwrittenFunction(Combine.onLoad, combineOnLoad)
